/*
 * main.c - args, configuration, logging
 *
 * trekhopd (Netrek firewall hop daemon)
 * By Andy McFadden
 */
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include "thd.h"


/*
 * Globals
 */
int verbose = FALSE;		/* if TRUE, print extra info on terminal */
char *prog = NULL;		/* our program name */
int listen_port = 6592;		/* where to listen */
int meta_listen_port;		/* where to listen for metaserver connection */
int approved_only = TRUE;	/* set from config file */
SERVER *approved = NULL;	/* list of approved servers */
int approved_count = 0;		/* #of servers in approved list */
int local_origin_only = TRUE;	/* set from config file */
OCTET_HOST *localnet = NULL;	/* list of local networks */
int localnet_count = 0;		/* #of networks in localnet list */


/*
 * Private variables
 */
static FILE *logfp = NULL;	/* log file */
static int do_log = TRUE;	/* set from config file */
static int truncate = 0;	/* truncate log file on startup? */


/*
 * Configuration stuff
 */
typedef enum {
    STMT_BOOL, STMT_INT, STMT_APPROVED, STMT_METASERVER, STMT_LOCALNET
} STMT_KIND;
struct keyword_t {
    STMT_KIND kind;
    char *keyword;
    int *var;
} keywords[] = {
    { STMT_INT,		"listen_port",	&listen_port },
    { STMT_INT, 	"meta_listen_port", &meta_listen_port },
    { STMT_BOOL,	"do_logging",	&do_log },
    { STMT_BOOL,	"approved_only", &approved_only },
    { STMT_BOOL,	"local_origin_only", &local_origin_only },
    { STMT_METASERVER,	"metaserver",	NULL },
    { STMT_APPROVED,	"approved",	NULL },
    { STMT_LOCALNET,	"localnet",	NULL },
};
#define MAX_KEYWORDS	(sizeof(keywords) / sizeof(struct keyword_t))


/*
 * print a message to the log
 *
 * (I'd use varargs, but vfprintf() doesn't exist everywhere, so why bother?)
 */
void
log_msg(format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
char *format;
int arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9;
{
    char buf[256];

    if (!do_log) return;
    if (logfp == NULL) {
	fprintf(stderr, "WARNING: attempt to write message to closed log\n");
	return;
    }

    /* date is in hex to keep it compact */
    sprintf(buf, "%.8lx: ", time(0));
    sprintf(buf+10, format, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7,
	arg8, arg9);
    if (*(buf + strlen(buf)-1) != '\n')
	strcat(buf, "\n");

    /* since there should only be one trekhopd process, we probably don't */
    /* need to seek to the end every time */
    fseek(logfp, (off_t) 0, 2);
    fprintf(logfp, buf);
}


/* add an entry to the "approved" list */
static void
add_approved(host, port)
char *host, *port;
{
    int portnum = atoi(port);

    /* don't allow approval of ports < 1024 EXCEPT port 592 */
    if (!portnum || (portnum < 1024 && portnum != 592)) {
	fprintf(stderr,"WARNING: can't approve port %d(%s); ignored\n",
		portnum, port);
	return;
    }

    if (approved == NULL) {
	approved = (SERVER *) malloc(sizeof(SERVER));
    } else {
	approved = (SERVER *) realloc(approved,
		sizeof(SERVER) * (approved_count+1));
    }
    if (approved == NULL) {
	fprintf(stderr, "ERROR: malloc/realloc failure (approved)\n");
	exit(1);
    }
    strcpy(approved[approved_count].hostname, host);
    approved[approved_count].port = portnum;
    approved_count++;
}


/* add an entry to the "localnet" list */
static void
add_localnet(addr)
char *addr;
{
    OCTET_HOST *lp;
    long na;

    if (localnet == NULL) {
	localnet = (OCTET_HOST *) malloc(sizeof(OCTET_HOST));
    } else {
	localnet = (OCTET_HOST *) realloc(localnet,
		sizeof(OCTET_HOST) * (localnet_count+1));
    }
    if (localnet == NULL) {
	fprintf(stderr, "ERROR: malloc/realloc failure (localnet)\n");
	exit(1);
    }
    if ((na = inet_network(addr)) == 0) {
        fprintf(stderr, "ERROR: localnet 0.0.0.0 is illegal\n");
	exit(1);
    }

    lp = &localnet[localnet_count++];
    lp->octet_addr[0] = (unsigned char) ( na        & 0x000000ff);
    lp->octet_addr[1] = (unsigned char) ((na >> 8 ) & 0x000000ff);
    lp->octet_addr[2] = (unsigned char) ((na >> 16) & 0x000000ff);
    lp->octet_addr[3] = (unsigned char) ((na >> 24) & 0x000000ff);

}


/* find the next token */
static char *
next_token(str)
char *str;
{
    /* skip leading whitespace */
    while (*str && (*str == ' ' || *str == '\t' || *str == '\n'))
	str++;
    if (*str)
	return (str);
    else
	return (NULL);
}


/*
 * read the configuration in (may be called from a signal handler)
 */
static void
load_config()
{
    FILE *fp;
    char buf[128], *cp, *cp2;
    int i, len, line;

    /* clear out any existing configuration stuff */
    if (approved_count) {
	free(approved);
	approved_count = 0;
    }
    if (localnet_count) {
	free(localnet);
	localnet_count = 0;
    }


    if ((fp = fopen(CONFIGFILE, "r")) == NULL) {
	perror("ERROR: can't open config file");
	exit(1);
    }

    line = 0;
    while (1) {
	fgets(buf, 128, fp);
	line++;
	if (feof(fp)) break;
	if (ferror(fp)) {
	    perror("ERROR: error reading config file");
	    exit(1);
	}
	if (buf[strlen(buf)-1] == '\n')
	    buf[strlen(buf)-1] = '\0';

	/* skip comments and blank lines */
	cp = next_token(buf);
	if (cp == NULL || *cp == '#')
	    continue;

	/* find a token in "keywords" which matches */
	for (i = 0; i < MAX_KEYWORDS; i++) {
	    len = strlen(keywords[i].keyword);	/* use table for speed? */
	    if (!strncmp(cp, keywords[i].keyword, len)) {
		cp2 = next_token(cp+len);
		if (cp2 != NULL && *cp2 == ':') {
		    break;
		}
	    }
	}
	if (i == MAX_KEYWORDS) {
	    fprintf(stderr, "WARNING: unknown statement (line %d): %s\n",
		line, buf);
	    continue;
	}

	/* parse the right-hand side with strtok */
	if ((cp = strtok(cp2+1, " \t")) == NULL) {
	    fprintf(stderr, "WARNING: missing rhs (line %d); ignored\n", line);
	    continue;
	}

	switch (keywords[i].kind) {
	case STMT_BOOL:		/* boolean on/off */
	    if (!strcasecmp(cp, "on"))
		*(keywords[i].var) = 1;
	    else if (!strcasecmp(cp, "off"))
		*(keywords[i].var) = 0;
	    else
		fprintf(stderr,
		    "WARNING: boolean expr must be 'on' or 'off' (line %d)\n",
		    line);
	    break;
	case STMT_INT:		/* integer value */
	    *((int *) keywords[i].var) = atoi(cp);
	    break;
	case STMT_APPROVED:	/* "approved" line */
	    if ((cp2 = strtok(NULL, " \t")) == NULL) {
		fprintf(stderr,
			"WARNING: missing port (line %d); ignored\n", line);
		break;
	    }
	    add_approved(cp, cp2);
	    break;
	case STMT_LOCALNET:	/* "localnet" line */
	    add_localnet(cp);
	    break;
	case STMT_METASERVER:	/* metaserver location */
	    if ((cp2 = strtok(NULL, " \t")) == NULL) {
		fprintf(stderr,
			"WARNING: missing port (line %d); ignored\n", line);
		break;
	    }
	    strcpy(meta_host, cp);
	    meta_port = atoi(cp2);
	    break;
	default:
	    /* "shouldn't happen" */
	    fprintf(stderr, "Internal error: bad keyword kind %d\n",
		keywords[i].kind);
	}
    }


    /*
     * sanity checks
     */
    if (approved_only && !approved_count) {
	fprintf(stderr, "ERROR: approved_only but no approved: lines\n");
	exit(1);
    }
    if (local_origin_only && !localnet_count) {
	fprintf(stderr, "ERROR: local_origin_only but no localnet: lines\n");
	exit(1);
    }
}


/*
 * open the log file and print a friendly message
 */
static void
open_log()
{
    time_t now;
    char *ostr;

    if (!do_log) return;

    /* prepare the log file */
    if (!truncate)
	ostr = "a";	/* just append */
    else
	ostr = "w";	/* truncate */

    if ((logfp = fopen(LOGFILE, ostr)) == NULL) {
	perror("ERROR: can't open log file for writing");
	exit(1);
    } else {
	setvbuf(logfp, NULL, _IONBF, 0);   /* no buffering */

	log_msg("\n");
	now = time(0);
	log_msg("** trekhopd v%s started on %s", VERSION, ctime(&now));
    }
}


/* fatal signal received */
static int
sig_death(sig)
int sig;
{
    if (verbose) printf("%s: killed by signal %d\n", prog, sig);
    log_msg("exit: killed by signal %d", sig);

    exit(2);
}


/* ejection signal receieved */
static int
sig_eject(sig)
int sig;
{
    if (verbose) printf("%s: received ejection signal %d\n", prog, sig);
    log_msg("DROPPING ALL CONNECTIONS (signal %d)", sig);

    kill_all = TRUE;
}


/* reconfigure signal receieved */
static int
sig_reconfig(sig)
int sig;
{
    if (verbose) printf("%s: received reconfigure signal %d\n", prog, sig);
    log_msg("rereading configuration file (signal %d)", sig);

    load_config();
}


/* put #of active connections in log */
/* (also puts info on active connections in USERFILE) */
static int
sig_count(sig)
int sig;
{
    if (verbose) printf("%s: rcvd count signal %d\n", prog, sig);
    log_msg("There are %d active connections", act_count());
}


/*
 * initialize signal handling stuff
 */
static void
init_signals()
{
    SIGNAL(SIGINT, sig_death);
    SIGNAL(SIGQUIT, sig_death);
    SIGNAL(SIGTERM, sig_death);
    SIGNAL(SIGHUP, sig_reconfig);
    SIGNAL(SIGUSR1, sig_eject);
    SIGNAL(SIGUSR2, sig_count);
    SIGNAL(SIGPIPE, SIG_IGN);
}


/*
 * Process args
 */
int
main(argc, argv)
int argc;
char *argv[];
{
    extern char *optarg;
    extern int optind;
    int c;

    /* parse args */
    while ((c = getopt(argc, argv, "vtp:")) != EOF)
	switch (c) {
	case 'v':
	    verbose++;
	    break;
	case 't':
	    truncate++;
	    break;
	case 'p':	/* overrides thdrc setting */
	    listen_port = atoi(optarg);
	    break;
	case '?':
	default:
	    fprintf(stderr, "Netrek firewall hop daemon v%s\n", VERSION);
	    fprintf(stderr, "usage: trekhopd [-v] [-t] [-p listen_port]\n\n");
	    exit(2);
	}

    if ((prog = strrchr(argv[0], '/')) == NULL)
	prog = argv[0];
    else
	prog++;

    /* read the configuration file */
    load_config();

    /* initialize internal data structures */
    init_connections();

    /* open the log and print a friendly message */
    if (do_log) open_log();

    /* init the signal handling stuff */
    init_signals();

    /* nothing has failed yet, so go wait for connections */
    main_loop();

    exit(0);
    /*NOTREACHED*/
}

