/*
 * meta.c - handle metaserver stuff (added as an afterthought)
 *
 * trekhopd (Netrek firewall hop daemon)
 * By Andy McFadden
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include "thd.h"

/*
 * globals
 */
int meta_port = 2520;			/* where to contact metaserver */
char *meta_host = "sickdog.cs.berkeley.edu";


/*
 * private stuff
 */
#define META_BUF_SIZE	4096	/* at most 4K of data */
static char metabuf[META_BUF_SIZE];
static int busy, csock, ssock;


/*
 * One goal of trekhopd is to avoid pure TCP passthru connections.  In
 * keeping with this goal, we handle metaserver queries by opening a connection
 * to the metaserver, getting all information, and then closing it before
 * passing the information on to the client.  This works perfectly well for
 * the metaserver but severely restricts attempts to use the metaserver
 * connection for other uses.
 */
int
connect_meta(fd)
int fd;
{
    struct sockaddr_in addr;
    struct hostent *hp;
    int length, flags;
    char *host;

    /*
     * this will prevent simultaneous connection attempts, but will cause
     * a poll for the duration of the first connection attempt.  The
     * alternative is to deny the second request totally or make it "sleep"
     * on the first.
     */
    if (busy) return (-2);

    /* accept connection from client */
    length = sizeof(struct sockaddr_in);
    if ((csock = accept(fd, &addr, &length)) < 0) {
	perror("WARNING: metaserver accept failed");
	return (-1);
    }

    /* figure out who we're connected to */
    length = sizeof(struct sockaddr_in);
    if (getpeername(csock, (struct sockaddr *) &addr, &length) < 0) {
	fprintf(stderr,
	    "WARNING: unable to get metaserver client peername; dropping\n");
	log_msg("%.2d refused: unable to get metaserver client peername",
		csock);
	close(csock);
	return (-1);
    }
    hp = gethostbyaddr((char *) &addr.sin_addr.s_addr, sizeof(long), AF_INET);
    if (hp != NULL)
	host = hp->h_name;
    else
	host = inet_ntoa(addr.sin_addr);

    /*
     * At this point we could do some validation of the source host (i.e.
     * make sure it's a localhost), but what's the point...?  Consider
     * "metaserver" lines in thdrc to be like "approved" lines, and don't
     * point them at internal hosts.
     */

    if (verbose) printf("metaserver request from %s (%d)\n", host,
	ntohs(addr.sin_port));
    log_msg("%.2d metaserver request from %s (%d)\n", csock, host,
	ntohs(addr.sin_port));

    /* establish TCP connection to metaserver */
    if ((ssock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	perror("socket (metaserver connection)");
	return (-1);
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(meta_port);
    if ((addr.sin_addr.s_addr = inet_addr(meta_host)) == -1) {
	if ((hp = gethostbyname(meta_host)) == NULL) {
	    log_msg("%.2d metaserver req denied: unknown host", csock);
	    close(ssock);
	    close(csock);
	    return (-1);
	} else {
	    addr.sin_addr.s_addr = *(long *) hp->h_addr;
	}
    }
    if ((flags = fcntl(ssock, F_GETFL, O_NDELAY)) < 0) {
	perror("unable to get flags for NB");
	return (-1);
    }
    if (fcntl(ssock, F_SETFL, flags|O_NDELAY) < 0) {
	perror("unable to set NB mode for connect");
	return (-1);
    }
    if (connect(ssock, &addr, sizeof(addr)) < 0) {
	if (errno != EINPROGRESS) {
	    perror("connect (metaserver connection)");
	    log_msg("%.2d connection to metaserver failed", csock);
	    close(ssock);
	    close(csock);
	    return (-1);
	}
    }

    busy = TRUE;
    return(ssock);		/* return server connection socket */
}


/*
 * Do the actual data transfer, now that connect() has completed.
 */
int
handle_meta()
{
    register int n, count;
    register char *buf;
    int flags;

    if (!busy) log_msg("meta busy bug!");
    busy = FALSE;

    /* clear NB mode */
    if ((flags = fcntl(ssock, F_GETFL, 0)) < 0) {
	perror("unable to get flags for NB clear");
	return (-1);
    }
    if (fcntl(ssock, F_SETFL, flags & (~O_NDELAY)) < 0) {
	perror("unable to clear NB mode for connect");
	return (-1);
    }

    count = 0;
    for (buf = metabuf ; count < META_BUF_SIZE ; buf += n) {
	if ((n = read(ssock, buf, META_BUF_SIZE-count)) <= 0) {
	    if (n < 0) {
		perror("metaserver read_data");
		log_msg("%.2d read error %d from metaserver", ssock, errno);
		close(ssock);
		close(csock);
		return (-1);
	    }
	    break;	/* EOF hit, metaserver is done sending */
			/* (can also happen if connect() failed) */
	}
	count += n;
    }
    if (n != 0) log_msg("NOTE: only partial metaserver data copied");

    close(ssock);	/* close metaserver connection */


    /* now send the data to the client, and close the connection */
    for (buf = metabuf ; count ; buf += n, count -= n) {
	if ((n = write(csock, buf, count)) < 0) {
	    perror("metaserver write_data");
	    log_msg("%.2d write error %d to metaserver query", csock, errno);
	    close(csock);
	    return (-1);
	}
    }

    close(csock);	/* close connection to client */
    log_msg("%.2d metaserver transfer complete", csock);
    return(0);
}

