/*
 * thd.c - the meaty part
 *
 * trekhopd (Netrek firewall hop daemon)
 * By Andy McFadden
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include "thd.h"

/*
 * globals
 */
int kill_all = FALSE;		/* set by signal handler */


/*
 * private stuff
 */
static int maxfd;

/* straight from packets.h */
typedef unsigned char uchar;
struct mesg_cpacket {
    char type;          /* CP_MESSAGE */
    char group;
    char indiv;
    char pad1;
    char mesg[80];
};
struct mesg_spacket {
    char type;          /* SP_MESSAGE */
    uchar m_flags;
    uchar m_recpt;
    uchar m_from;
    char mesg[80];
};


/*
 * There will be one SESSION per netrek client, and four different file
 * descriptors (two TCP, two UDP) will point to the CONNECTIONs within it.
 */
typedef enum { CONN_NULL, CONN_TCP, CONN_SRV_UDP, CONN_CLI_UDP } CONN_KIND;
typedef struct {
    CONN_KIND kind;
    int   sock;
    struct session_t *session;
} CONNECTION;

#define MAXHOSTLEN	64
#define MAXUSERLEN	8
typedef enum { ST_CLOSED, ST_OPEN, ST_MAGIC_OK, ST_CONNECTING, ST_CONNECTED }
    SESS_STATE;
typedef struct session_t {
    SESS_STATE state;
    char  user[MAXUSERLEN];
    char  client_host[MAXHOSTLEN];
    long  client_addr;
    char  server_host[MAXHOSTLEN];
    long  server_addr;
    int   tcp_client_port;
    int   tcp_server_port;
    int   udp_client_port;
    int   udp_server_port;
    struct mesg_cpacket initial_buf;
    int   initial_count;

    CONNECTION tcp_client;
    CONNECTION tcp_server;
    CONNECTION udp_client;
    CONNECTION udp_server;

    /* hack to handle serv_port correctly */
    int   serv_port;
    int   gw_port;
    int   gw_serv_port;

    /* statistics */
    time_t connect_when;
    time_t udp_when;
    long  tcp_out_total;	/* #of bytes; out means client->server */
    long  tcp_out_count;	/* #of packets */
    long  tcp_in_total;		/* in means server->client */
    long  tcp_in_count;
    long  udp_out_total;
    long  udp_out_count;
    long  udp_in_total;
    long  udp_in_count;
} SESSION;


/*
 * private variables
 */
static CONNECTION **fdtab = NULL;
fd_set readfd_set, writefd_set;

#define RDBUF_SIZE	32767
static char rdbuf[RDBUF_SIZE];


#define MAGIC	1		/* SP_MESSAGE */
#define MAGIC_SIZE	1	/* one byte */

/*
 * Allocate / initialize local data structures
 */
void
init_connections()
{
    /* this system call may not be portable to non-BSD; try FD_SETSIZE */
    maxfd = getdtablesize();
    if (maxfd > FD_SETSIZE)
	maxfd = FD_SETSIZE;
    if (verbose) printf("%s: maximum file descriptor is %d\n", prog, maxfd);

    /* allocate and init to NULL the per-fd pointers */
    fdtab = (CONNECTION **) malloc(sizeof(CONNECTION *) * maxfd);
    bzero(fdtab, sizeof(CONNECTION *) * maxfd);
}


/*
 * Alloc a new SESSION.
 */
static SESSION *
alloc_session()
{
    SESSION *news;

    if ((news = (SESSION *) malloc(sizeof(SESSION))) == NULL) {
	if (verbose) fprintf(stderr, "ERROR: malloc failed in alloc_session\n");
	log_msg("exit: malloc failure");
	exit(1);
    }
    news->state = ST_CLOSED;
    news->tcp_client_port = news->tcp_server_port = -1;
    news->udp_client_port = news->udp_server_port = -1;
    news->initial_count = 0;

    news->tcp_client.kind = CONN_TCP;
    news->tcp_client.sock = -1;
    news->tcp_client.session = news;
    news->tcp_server.kind = CONN_TCP;
    news->tcp_server.sock = -1;
    news->tcp_server.session = news;
    news->udp_client.kind = CONN_CLI_UDP;
    news->udp_client.sock = -1;
    news->udp_client.session = news;
    news->udp_server.kind = CONN_SRV_UDP;
    news->udp_server.sock = -1;
    news->udp_server.session = news;

    /* probably ought to just bzero the damn thing */
    news->connect_when = 0;
    news->udp_when = 0;
    news->tcp_out_total = 0;
    news->tcp_out_count = 0;
    news->tcp_in_total = 0;
    news->tcp_in_count = 0;
    news->udp_out_total = 0;
    news->udp_out_count = 0;
    news->udp_in_total = 0;
    news->udp_in_count = 0;

    return (news);
}


/*
 * Kill a session; close all descriptors and clean up
 */
#define NUKE(x) { \
    if ((x)->sock != -1) { \
	FD_CLR((x)->sock, &readfd_set); fdtab[(x)->sock] = NULL; \
	close((x)->sock); (x)->sock = -1; (x)->kind = CONN_NULL; \
    } \
}
static void
kill_session(sp)
SESSION *sp;
{
    if (sp->tcp_client.sock == -1) {
	if (verbose) printf("bug!\n");
	log_msg("BUG: kill_session with sock == -1");
	log_msg("     session state is %d", sp->state);
	sleep(1);
	return;		/* bug */
    }
    if (verbose) printf("%s: killing %d\n", prog, sp->tcp_client.sock);
    log_msg("%.2d closed (%d %d %d) after %d seconds (%d since UDP)",
	sp->tcp_client.sock, sp->tcp_server.sock,
	sp->udp_client.sock, sp->udp_server.sock,
	time(0) - sp->connect_when,
	(sp->udp_when != (time_t)0) ? time(0) - sp->udp_when : 0);
    log_msg("%.2d tcp out: %ld/%ld in: %ld/%ld  udp out: %ld/%ld in: %ld/%ld",
	sp->tcp_client.sock,
	sp->tcp_out_total, sp->tcp_out_count,
	sp->tcp_in_total,  sp->tcp_in_count,
	sp->udp_out_total, sp->udp_out_count,
	sp->udp_in_total,  sp->udp_in_count);

    switch (sp->state) {
    case ST_CLOSED:
	break;
    case ST_OPEN:
    case ST_MAGIC_OK:
    case ST_CONNECTING:
	/* only tcp_client should be valid, but we could be between states */
    case ST_CONNECTED:
	NUKE(&(sp->tcp_client));
	NUKE(&(sp->tcp_server));
	NUKE(&(sp->udp_client));
	NUKE(&(sp->udp_server));
	break;
    default:
	if (verbose)
	    fprintf(stderr, "Internal error: bad state in kill_session (%d)\n",
		sp->state);
	log_msg("BUG: bad state %d in kill_session", sp->state);
	break;
    }

    free(sp);
}


/*
 * New connection
 *
 * Note that we don't open the connection to the server yet, since we don't
 * know where the server is.
 */
static void
new_connection(sock)
int sock;
{
    SESSION *news;
    OCTET_HOST oh, *lp;
    struct sockaddr_in sin;
    struct hostent *hp;
    time_t t;
    int length, port;
    int i, j, newsock;
    char *cp, *host;

    log_msg("+new_connection(%d)\n", sock);	/* debugging */

    /* accept the new connection */
    /* (could make this nonblocking as well, but that's too painful) */
    length = sizeof(struct sockaddr_in);
    if ((newsock = accept(sock, &sin, &length)) < 0) {
	int err = errno;
	if (verbose) perror("WARNING: accept failed");
	log_msg("accept() failed, err=%d\n", err);
	return;
    }

    /* figure out who we're connected to */
    length = sizeof(struct sockaddr_in);
    if (getpeername(newsock, (struct sockaddr *) &sin, &length) < 0) {
	if (verbose) fprintf(stderr,
		"WARNING: unable to get client peername; dropping him\n");
	log_msg("%.2d refused: unable to get client peername", newsock);
	close(newsock);
	return;
    }
    hp = gethostbyaddr((char *) &sin.sin_addr.s_addr, sizeof(long), AF_INET);
    if (hp != NULL)
	host = hp->h_name;
    else
	host = inet_ntoa(sin.sin_addr);
    port = ntohs(sin.sin_port);


    /* validate the source host against localnet */
    oh.octet_addr[0] =(unsigned char)( sin.sin_addr.s_addr        & 0x000000ff);
    oh.octet_addr[1] =(unsigned char)((sin.sin_addr.s_addr >> 8 ) & 0x000000ff);
    oh.octet_addr[2] =(unsigned char)((sin.sin_addr.s_addr >> 16) & 0x000000ff);
    oh.octet_addr[3] =(unsigned char)((sin.sin_addr.s_addr >> 24) & 0x000000ff);

    lp = localnet;	/* don't forget, this can be NULL */
    for (i = 0; i < localnet_count; i++, lp++) {
	for (j = 0; j < 4; j++) {
	    if (!lp->octet_addr[j]) continue;
	    if (lp->octet_addr[j] != oh.octet_addr[j]) break;
	}
	if (j == 4) break;		/* we matched! */
    }
    if (localnet_count && (i == localnet_count)) {
	if (local_origin_only) {
	    log_msg("%.2d REFUSING connection from %s (%d)", newsock, host,
		port);
	    close(newsock);
	    return;
	} else {
	    log_msg("%.2d accepting NON-LOCAL from %s (%d)", newsock,
		host, port);
	}
    }
    t = time(0);
    cp = ctime(&t);
    *(cp+20) = '\0';	/* this may not be too bright */
    log_msg("%.2d connection from %s (%d) %s", newsock, host, port, cp);


    /* alloc and initialize new SESSION and CONNECTIONs */
    news = alloc_session();
    news->state = ST_OPEN;
    news->connect_when = t;		/* record connection time */

    strncpy(news->client_host, host, MAXHOSTLEN);
    news->client_addr = sin.sin_addr.s_addr;
    news->tcp_client_port = port;
    news->tcp_client.sock = newsock;

    /* set up this fd, so info gets sent to TCP server port */
    fdtab[newsock] = &(news->tcp_server);

    FD_SET(newsock, &readfd_set);

    if (verbose) {
	printf("%s: %.2d: new connection from %s(%d)\n", prog, newsock,
		host, port);
    }
}


/*
 * validate the client's request, and connect him to the server
 */
static int
connect_server(sp, statep)
SESSION *sp;
SESS_STATE *statep;
{
    OCTET_HOST oh, *lp;
    struct sockaddr_in addr;
    struct hostent *hp;
    struct mesg_cpacket *mp;
    int *ip, port_req, gw_serv_port, gw_port, gw_local_port;
    char *userid, *host;
    int i, j, sock, usock, idsock, len;
    int flags;

    /* pull stuff out of the packet */
    mp = &(sp->initial_buf);
    ip = (int *) mp->mesg;

    port_req = ntohl(*ip++);
    gw_local_port = ntohl(*ip++);
    userid = (char *) ip;
    host = ((char *) ip) + MAXUSERLEN;

    idsock = sp->tcp_client.sock;	/* used for log messages */
    log_msg("%.2d %.8s --> %s %d (client UDP:%d)", idsock, userid,
	host, port_req, gw_local_port);

    /* do the host lookup (need for connection and for some checks) */
    if ((addr.sin_addr.s_addr = inet_addr(host)) == -1) {
	if ((hp = gethostbyname(host)) == NULL) {
	    log_msg("%.2d DENIED: unknown host", idsock);
	    /*close(sock);*/
	    return (-1);
	} else {
	    addr.sin_addr.s_addr = *(long *) hp->h_addr;
	}
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(port_req);

    /* see if it's on the approved list (overrides default checks) */
    /* (might make sense to convert to numeric first, but not needed) */
    for (i = 0; i < approved_count; i++) {
	if (!strcasecmp(host, approved[i].hostname) &&
		port_req == approved[i].port) {
	    goto approved;	/* note this skips all other validity checks */
	}
    }

    if (approved_only) {
	log_msg("%.2d DENIED: host/port not on approved list", idsock);
	return (-1);
    }

    /* validate the destination */
    if (port_req < IPPORT_RESERVED) {
	log_msg("%.2d DENIED: privileged port", idsock);
	return (-1);
    }

    /* validate the server host against localnet (should NOT match) */
    oh.octet_addr[0] =(unsigned char)( addr.sin_addr.s_addr       & 0x000000ff);
    oh.octet_addr[1] =(unsigned char)((addr.sin_addr.s_addr >>8 ) & 0x000000ff);
    oh.octet_addr[2] =(unsigned char)((addr.sin_addr.s_addr >>16) & 0x000000ff);
    oh.octet_addr[3] =(unsigned char)((addr.sin_addr.s_addr >>24) & 0x000000ff);

    lp = localnet;	/* don't forget, this can be NULL */
    for (i = 0; i < localnet_count; i++, lp++) {
	for (j = 0; j < 4; j++) {
	    if (!lp->octet_addr[j]) continue;
	    if (lp->octet_addr[j] != oh.octet_addr[j]) break;
	}
	if (j == 4) break;		/* we matched! */
    }
    if (localnet_count && (i != localnet_count)) {
	/* we matched a local network; reject it */
	log_msg("%.2d DENIED: host on localnet list", idsock);
	return (-1);
    }

approved:
    /* establish TCP connection to server */
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	log_msg("socket(server) failed, err=%d", errno);
	return (-1);
    }

    if ((flags = fcntl(sock, F_GETFL, 0)) < 0) {
	log_msg("fcntl(F_GETFL) failed, err=%d", errno);
	return(-1);
    }
    if (fcntl(sock, F_SETFL, flags|O_NDELAY) < 0) {
	log_msg("fcntl(F_SETFL-delay) failed, err=%d", errno);
	return (-1);
    }
    if (connect(sock, &addr, sizeof(addr)) < 0) {
	if (errno != EINPROGRESS) {
	    log_msg("conect(server) failed, err=%d", errno);
	    close(sock);
	    return (-1);
	} else {
	    /* connect has not taken effect yet */
	    /* (clear the client readfd so we don't poll and nuke the CPU) */
	    FD_SET(sock, &writefd_set);
	    FD_CLR(sp->tcp_client.sock, &readfd_set);
	    *statep = ST_CONNECTING;
	}
    } else {
	/* connect succeeded immediately */
	FD_SET(sock, &readfd_set);
	*statep = ST_CONNECTED;
    }

    /* we're set; hook it in */
    sp->tcp_server.kind = CONN_TCP;
    sp->tcp_server.sock = sock;
    strncpy(sp->server_host, host, MAXHOSTLEN);
    strncpy(sp->user, userid, MAXUSERLEN);
    sp->server_addr = addr.sin_addr.s_addr;
    fdtab[sock] = &(sp->tcp_client);


    /* now do UDP if requested */
    if (gw_local_port) {
	/*
	 * we listen for a connection from the server on gw_serv_port
	 * we listen for a connection from the client on gw_port
	 * we connect to the client host on gw_local_port
	 */
	/* prepare connection to server */
	if ((usock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	    log_msg("socket(DGRAM-1) failed, err=%d", errno);
	    return (-1);
	}
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	addr.sin_port = 0;
	if (bind(usock, &addr, sizeof(addr)) < 0) {
	    log_msg("%.2d unable to bind(1), err=%d", usock, errno);
	    close(usock);
	    return (-1);
	}
	len = sizeof(addr);
	if (getsockname(usock, &addr, &len) < 0) {
	    log_msg("getsockname(1) failed, err=%d\n", errno);
	    close(usock);
	    return (-1);
	}
	gw_serv_port = ntohs(addr.sin_port);

	/* (no connect yet; we don't know server's port) */
	sp->udp_server.kind = CONN_SRV_UDP;
	sp->udp_server.sock = usock;
	fdtab[usock] = &(sp->udp_client);
	FD_SET(usock, &readfd_set);
	if (verbose) {
	    log_msg("connection to server:");
	    printUdpInfo(usock);
	}

	/* prepare connection to client */
	if ((usock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	    log_msg("socket(DGRAM-2) failed, err=%d\n", errno);
	    return (-1);
	}
	addr.sin_port = 0;
	if (bind(usock, &addr, sizeof(addr)) < 0) {
	    log_msg("%.2d unable to bind(2), err=%d", usock, errno);
	    close(usock);
	    return (-1);
	}
	len = sizeof(addr);
	if (getsockname(usock, &addr, &len) < 0) {
	    log_msg("getsockname(2) failed, err=%d\n", errno);
	    close(usock);
	    return (-1);
	}
	gw_port = ntohs(addr.sin_port);

	addr.sin_addr.s_addr = sp->client_addr;
	addr.sin_port = htons(gw_local_port);
	if (connect(usock, &addr, sizeof(addr)) < 0) {
	    log_msg("%.2d UDP connect failed, err=%d", usock, errno);
	    close(usock);
	    return (-1);
	}
	sp->udp_client.kind = CONN_CLI_UDP;
	sp->udp_client.sock = usock;
	fdtab[usock] = &(sp->udp_server);
	FD_SET(usock, &readfd_set);

	if (verbose) {
	    log_msg("connection to client:");
	    printUdpInfo(usock);
	}
    }

    log_msg("%.2d +client %-2d  server %-2d  udp client %-2d  udp server %-2d",
	idsock,
	sp->tcp_client.sock, sp->tcp_server.sock,
	sp->udp_client.sock, sp->udp_server.sock);

    /* now send a response packet back to the client */
    sp->gw_port = gw_port;
    sp->gw_serv_port = gw_serv_port;

    return (0);
}


/*
 * When the non-blocking server connection request completes, we get here.
 * If the connection attempt failed, then we will get an error the first
 * time we try to write (SIGPIPE too).  We can clean up then.
 */
static void
server_connected(sock)
{
    SESSION *sp;
    struct sockaddr_in saddr;
    struct mesg_spacket reply;
    int idsock, flags;
    int len, serv_port;
    long *lp;

    if (fdtab[sock] == NULL) {
	log_msg("ERROR null in server_connected");
	/*abort();*/
	return;
    }

    sp = fdtab[sock]->session;
    idsock = sp->tcp_client.sock;

    FD_CLR(sock, &writefd_set);			/* no longer needed */
    FD_SET(sock, &readfd_set);			/* add TCP server */
    FD_SET(sp->tcp_client.sock, &readfd_set);	/* re-enable TCP client */
    sp->state = ST_CONNECTED;			/* ready to rock */

    /* clear non-blocking mode */
    if ((flags = fcntl(sock, F_GETFL, 0)) < 0) {
	if (verbose) perror("unable to get flags for NB clear");
	log_msg("ERROR: NB clear F_GETFL failed for %d, err=%d", sock, errno);
    } else {
	if (fcntl(sock, F_SETFL, flags & (~O_NDELAY)) < 0) {
	    if (verbose) perror("unable to clear NB mode after connect");
	    log_msg("ERROR: NB reset F_SETFL failed for %d, err=%d",sock,errno);
	}
    }

    /* form up and send the reply */
    len = sizeof(saddr);
    if (getpeername(sock, &saddr, &len) < 0) {
	/* will get an ENOTCONN if the connect() failed; no big deal */
	if (errno == ENOTCONN)
	    log_msg("%.2d getpeername() - earlier connect() failed", sock);
	else
	    log_msg("ERROR: getpeername(%d): err %d", sock, errno);
        serv_port = -1;
    } else {
	serv_port = ntohs(saddr.sin_port);
    }
    reply.type = 1;	/* SP_MESSAGE */
    reply.m_flags = reply.m_recpt = reply.m_from = 0;
    lp = (long *) reply.mesg;
    *lp++ = htonl(sp->gw_serv_port);
    *lp++ = htonl(sp->gw_port);
    *lp++ = htonl(serv_port);
    if (write_data(sp->tcp_client.sock, &reply, sizeof(reply)) < 0) {
	log_msg("%.2d unable to send reply to client", sp->tcp_client.sock);
	/* ack!  keep going, maybe nobody will notice */
    }

    if (verbose) printf("%s: connected %d\n", prog, idsock);
    log_msg("%.2d connected", idsock);
}


/*
 * debug
 */
printUdpInfo(sock)
int sock;
{
    struct sockaddr_in addr;
    int len;

    len = sizeof(addr);
    if (getsockname(sock, &addr, &len) < 0) {
/*      if (verbose) perror("printUdpInfo: getsockname");*/
        return;
    }
    log_msg("+ LOCAL: addr=0x%x, family=%d, port=%d\n", addr.sin_addr.s_addr,
        addr.sin_family, ntohs(addr.sin_port));

    if (getpeername(sock, &addr, &len) < 0) {
/*      if (verbose) perror("printUdpInfo: getpeername");*/
        return;
    }
    log_msg("+ PEER : addr=0x%x, family=%d, port=%d\n", addr.sin_addr.s_addr,
        addr.sin_family, ntohs(addr.sin_port));
}


/*
 * Read from UDP server, updating data structures if the port changes.
 *
 * NOTE: had a really strange bug here on charon where trekhopd would
 * get data from charon's port 53 (domain).  What the hell?  Quick fix
 * is to prevent trekhopd from switching to new host/port if the port
 * number is < 1024.
 */
static int
read_udp_server(sp)
SESSION *sp;
{
    struct sockaddr_in from;
    struct hostent *hp;
    int cc, usock;
    int length;


    usock = sp->udp_server.sock;
    length = sizeof(from);
    if ((cc = recvfrom(usock, rdbuf, RDBUF_SIZE, 0, &from, &length)) <= 0) {
	if (verbose) perror("bad recvfrom");
	log_msg("%.2d bad UDP recvfrom", usock);
	return (-1);
    }

    if (sp->server_addr != from.sin_addr.s_addr) {
	/* this is the "calvin" bug - advertised IP addr != from.addr */
	if (verbose) printf("%.2d strange UDP host addr\n", usock);
	log_msg("%.2d note: UDP from 0x%.8lx, expected from 0x%.8lx\n",
	    usock, from.sin_addr.s_addr, sp->server_addr);
	/* check for weird charon glitch */
	if (ntohs(from.sin_port) < 1024) {
	    log_msg("%.2d WHOA: bogus source address, addr not changed", usock);
	    return (0);
	}
	sp->server_addr = from.sin_addr.s_addr;
    }

    if (ntohs(from.sin_port) != sp->udp_server_port) {
	/* new connection or we just got reconnected */
	hp = gethostbyaddr((char *) &from.sin_addr.s_addr,sizeof(long),AF_INET);
	if (sp->udp_server_port == -1) {
	    if (hp != NULL)
		log_msg("%.2d UDP from server %s", usock, hp->h_name);
	    else
		log_msg("%.2d UDP from server %s", usock,
						    inet_ntoa(from.sin_addr));
	}

	sp->udp_server_port = ntohs(from.sin_port);

	log_msg("%.2d UDP connection complete to server port %d", usock,
	    sp->udp_server_port);

	if (sp->udp_when == (time_t) 0)
	    sp->udp_when = time(0);
    }

    return (cc);
}


/*
 * Send data to UDP server
 */
static int
write_udp_server(sp, buf, count)
SESSION *sp;
register char *buf;
register int count;
{
    struct sockaddr_in addr;
    long length, orig = count;
    register int n;

    /* now send it */
    addr.sin_family = AF_INET;
    addr.sin_port = sp->udp_server_port;
    addr.sin_addr.s_addr = sp->server_addr;
    length = sizeof(addr);
    for ( ; count ; count -= n, buf += n) {
	if ((n = sendto(sp->udp_server.sock, buf, count, 0, &addr, length))<0) {
	    if (verbose) perror("sendto");
	    log_msg("%.2d UDP sendto, err=%d", sp->udp_server.sock, errno);
	    return (-1);
	}
    }

    return (orig);
}


/*
 * read a bunch of data
 *
 * Return value is #of bytes read, which will be > 0 on success, -1 on error,
 * or -2 eof hit.
 */
static int
read_data(fd, buf, count)
int fd;
char *buf;
int count;
{
    int cc;

    if (!count) {
	count = RDBUF_SIZE;
	buf = rdbuf;
    }
    if ((cc = read(fd, buf, count)) <= 0) {
	if (!cc) return (-2);	/* EOF */
	if (verbose) perror("WARNING: unable to read");
	log_msg("%.2d read error %d", fd, errno);
	return (-1);
    }
    return (cc);
}


/*
 * write a bunch of data
 */
static int
write_data(fd, buf, count)
int fd;
register char *buf;
register int count;
{
    register long n;
    int orig = count;

    for (; count ; buf += n, count -= n) {
	if ((n = write(fd, buf, count)) < 0) {
	    if (verbose) perror("write_data");
	    if (errno == EPIPE)
		log_msg("%.2d unable to connect (EPIPE)", fd);
	    else
		log_msg("%.2d write error %d", fd, errno);
	    return (-1);
	}
    }
    return (orig);
}


/*
 * Handle a descriptor which select() claims has activity on it.  This does
 * not handle new connections.
 *
 * Two reasons for the apparent complexity: (1) we don't want to hang while
 * waiting for a client to complete the opening sequence, and (2) we can't
 * fully establish the UDP link until the server sends us a UDP packet.  Note
 * also that a UDP connection may open and close several times during the
 * lifetime of a TCP connection; the server's port # can change each time.
 */
static void
handle(fd)
int fd;
{
    CONNECTION *conp;
    SESS_STATE state;
    SESSION *sp;
    char *bp;
    int cc;

    /* this happens when we have data on 7 and 8 and we close 7 */
    if (fdtab[fd] == NULL) return;

    /* shortcuts */
    conp = fdtab[fd];
    sp = conp->session;
    state = sp->state;
    bp = (char *) &(sp->initial_buf);

    if (conp->kind == CONN_NULL) {
	/* probably activity on a now-dead fd; the printf() is used so we */
	/* can tell if we're spinning (besides the huge increase in CPU!) */
	if (verbose) printf("tried to work on NULL fd %d\n", fd);
	return;
    }
    if (state != ST_CONNECTED && conp->kind != CONN_TCP) {
	if (verbose) fprintf(stderr,
		"Internal error: non-TCP on unconnected fd %d\n", fd);
	log_msg("BUG: non-TCP on unconnected fd %d\n", fd);
	kill_session(sp);
	return;
    }

    /*if (conp->kind != CONN_TCP) printf("activity on %d\n", fd); /*debug*/

    switch (conp->kind) {
    case CONN_NULL:
	if (verbose) fprintf(stderr,
		"WARNING: received activity on inactive fd %d\n", fd);
	log_msg("NOTE: received activity on inactive fd %d\n", fd);
	break;
    case CONN_TCP:
	switch (state) {
	case ST_CLOSED:
	    if (verbose) fprintf(stderr,
		"Internal error: TCP on ST_CLOSED socket %d\n", fd);
	    log_msg("NOTE: TCP on ST_CLOSED socket %d\n", fd);
	    break;
	case ST_OPEN:
	    if ((cc = read_data(fd, bp + sp->initial_count,
		    MAGIC_SIZE - sp->initial_count)) < 0) {
		log_msg("NOTE: unable to read magic; killing %d\n", fd);
		kill_session(sp);
		return;
	    }
	    sp->initial_count += cc;
	    if (sp->initial_count == MAGIC_SIZE) {
		if (*bp == MAGIC) {
#ifdef DEBUG
		    if (verbose) printf("%s: magic ok on %d\n", prog, fd);
#endif
		    sp->state = ST_MAGIC_OK;
		} else {
		    if (verbose) {
			printf("%s: magic failed on %d, dropping\n", prog, fd);
			printf("got %d, wanted %d\n", *bp, MAGIC);
		    }
		    log_msg("NOTE: got bad magic on %d; killing\n", fd);
		    kill_session(sp);
		    return;
		}
	    }
	    break;
	case ST_MAGIC_OK:
	    if ((cc = read_data(fd, bp + sp->initial_count,
		    sizeof(struct mesg_cpacket) - sp->initial_count)) < 0) {
		if (verbose) fprintf(stderr,
			"WARNING: unable to read mesg; killed %d\n",fd);
		log_msg("NOTE: unable to read msg; killing %d\n", fd);
		kill_session(sp);
		return;
	    }
	    sp->initial_count += cc;
	    if (sp->initial_count == sizeof(struct mesg_cpacket)) {
		if (connect_server(sp, &(sp->state)) < 0) {
		    log_msg("%.2d connect_server failed, killing session", fd);
		    kill_session(sp);
		} else {
		    /* connect_server() sets sp->state to CONNECTING or -ED */
		}
	    }
	    break;
	case ST_CONNECTING:
	    /* still waiting for server connection */
	    /* (this may burn some CPU if data is available to read) */
	    break;
	case ST_CONNECTED:
	    /* this is the usual case; just copy */
	    if ((cc = read_data(fd, NULL, 0)) < 0) {
		/* read a whole buffer full */
		if (cc == -2)
		    log_msg("%.2d EOF hit, killing session", fd);
		else
		    log_msg("%.2d read_data failed, killing session", fd);
		kill_session(sp);
	    } else {
		if (write_data(conp->sock, rdbuf, cc) < 0) {
		    log_msg("%.2d write_data failed, killing session",
			conp->sock);
		    kill_session(sp);
		}
		if (fd == sp->tcp_client.sock) {
		    sp->tcp_out_total += cc;	/* from client to server */
		    sp->tcp_out_count++;
		} else {
		    sp->tcp_in_total += cc;	/* from server to client */
		    sp->tcp_in_count++;
		}
	    }
	    break;
	default:
	    if (verbose) fprintf(stderr,
		"Internal error: bogus ss->state in handle(%d)\n", fd);
	    log_msg("NOTE: bad ss->state in handle(%d) (%d)", fd, state);
	    break;
	}

	break;		/* end of CONN_TCP */

    /*
     * Thought for the day: it may be possible for us to end up blocking
     * on a read somewhere, which would cause the hangs seen periodically
     * after network or machine failures.  Need to improve the error
     * handling somehow, or have some sort of "ghostbust" detection that
     * automatically fries everybody if the program seems to be stuck.
     */

    case CONN_SRV_UDP:
	/* client-->server path never connected, so use sendto() instead */
	if ((cc = read_data(fd, NULL, 0)) < 0) {
	    log_msg("%.2d UDP cread=%d\n", fd, cc);
	} else {
	    /*printf("srv_udp: got %d bytes\n", cc);	/* debug */
	    if ((cc = write_udp_server(sp, rdbuf, cc)) < 0) {
		if (verbose) printf("%.2d UDP closed (wc)\n", fd);
		log_msg("%.2d UDP swrite=%d\n", fd, cc);
		/* do something else? */
	    } else {
		sp->udp_out_total += cc;
		sp->udp_out_count++;
	    }
	}
	break;
    case CONN_CLI_UDP:
	/* server-->client path always set, but do recvfrom() to get port */
	if ((cc = read_udp_server(sp)) <= 0) {
	    if (!cc) break;	/* weird charon glitch */
	    if (verbose) printf("%.2d UDP closed (s)\n", fd);
	    log_msg("%.2d UDP sread=%d\n", fd, cc);
	} else {
	    /*printf("cli_udp: got %d bytes\n", cc);	/* debug */
	    if ((cc = write_data(conp->sock, rdbuf, cc)) < 0) {
		if (verbose) printf("%.2d UDP closed (wc)\n", fd);
		log_msg("%.2d UDP cwrite=%d\n", fd, cc);
		/* do something else? */
	    } else {
		sp->udp_in_total += cc;
		sp->udp_in_count++;
	    }
	}
	break;
    default:
	if (verbose) fprintf(stderr,
		"Internal error: bogus conp->kind in handle(%d)\n", fd);
	log_msg("NOTE: bad conp->kind in handle(%d) (%d)", fd, conp->kind);
	break;
    }
}


/*
 * return the #of active connections
 *
 * now also creates or overwrites USERFILE with active connection info
 */
int
act_count()
{
    FILE *fp;
    SESSION *sp;
    int i, count;

    if ((fp = fopen(USERFILE, "w")) == NULL)
	log_msg("Unable to open %s (err %d)", USERFILE, errno);

    for (i = count = 0; i < maxfd; i++) {
	if (fdtab[i] == NULL) continue;
	sp = fdtab[i]->session;
	if (fdtab[i]->sock == sp->tcp_client.sock) {
	    count++;
	    if (fp != NULL) {
		fprintf(fp, "%.2d %.8s: %.64s --> %.64s\n",
		    sp->tcp_client.sock, sp->user,
		    sp->client_host, sp->server_host);
	    }
	}
    }
    if (fp != NULL)
	fclose(fp);
    log_msg("Active user list placed in %s", USERFILE);

    return (count);
}


/*
 * main program loop
 */
void
main_loop()
{
    register int i;
    struct sockaddr_in addr;
    int on = 1;
    fd_set readfds, writefds;
    int cc, mcc, lsock, mlsock, msock;

    /*
     * Prepare socket to accept calls
     */
    if ((lsock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	if (verbose) perror("ERROR: unable to create socket to listen to");
	/* perror() could wipe errno, but if you're -v then you've already
	   seen the error, so you don't need it in the log file.  laziness++ */
	log_msg("exit: unable to create listen socket, err=%d", errno);
	exit(1);
    }
    setsockopt(lsock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port = htons(listen_port);
    if (bind(lsock, &addr, sizeof(addr)) < 0) {
	if (verbose) perror("ERROR: unable to bind to socket");
	log_msg("exit: unable to bind to listen socket, err=%d", errno);
	exit(1);
    }
    if (listen(lsock, 5) < 0) {
	if (verbose) perror("ERROR: unable to listen to socket\n");
	log_msg("exit: unable to listen to listen socket, err=%d", errno);
	exit(1);
    }

    /* init the fd set */
    FD_ZERO(&readfd_set);
    FD_ZERO(&writefd_set);
    FD_SET(lsock, &readfd_set);
    if (verbose) printf("%s: Listening on port %d\n", prog, listen_port);
    log_msg("Listening on port %d, fd %d\n", listen_port, lsock);


    /* metaserver stuff */
    if ((mlsock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	if (verbose) perror("ERROR: unable to create socket to listen to");
	log_msg("WARNING: unable to create listen socket");
    } else {
	setsockopt(mlsock, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	addr.sin_port = htons(meta_listen_port);
	if (bind(mlsock, &addr, sizeof(addr)) < 0) {
	    if (verbose) perror("ERROR: unable to bind to socket");
	    log_msg("WARNING: unable to bind to meta socket, err=%d", errno);
	}
	if (listen(mlsock, 5) < 0) {
	    if (verbose) perror("ERROR: unable to listen to socket\n");
	    log_msg("WARNING: unable to listen to meta socket, err=%d", errno);
	}
	FD_SET(mlsock, &readfd_set);
	if (verbose) printf("%s: metaserver listen on port %d\n", prog,
		meta_listen_port);
    }
    msock = -1;

    while (1) {
	bcopy(&readfd_set, &readfds, sizeof(fd_set));
	bcopy(&writefd_set, &writefds, sizeof(fd_set));
	if (!FD_ISSET(lsock, &readfds)) {
	    /* bug! */
	    log_msg("WARNING: lsock not in readfds\n");
	    FD_SET(lsock, &readfd_set);
	}
	if ((cc = select(maxfd, &readfds, &writefds, NULL, NULL)) <= 0) {
	    if (cc < 0) {
		if (verbose) {
		    fprintf(stderr, "%s: ", prog);
		    perror("select");
		}
		if (errno != EINTR) {
		    log_msg("select() failed, err=%d\n", errno);
		    sleep(1);	/* don't make a huge log file with errno=9 */
		}
		/* usually an interrupted system call */
	    }
	    if (!kill_all) continue;	/* kill_all signal was cause */
	}
	if (kill_all) {
	    log_msg("-- killing all sessions");
	    /* kill all sessions */
	    for (i = 0; i < maxfd; i++) {
		if (fdtab[i] == NULL) continue;
		if (fdtab[i]->sock >= 0)
		    kill_session(fdtab[i]->session);
	    }
	    kill_all = FALSE;
	    continue;
	}
	if (lsock != 5) log_msg("+bad");	/* watch for bug */

	/* cc is #of interesting file descriptors */
	for (i = 0; i < maxfd && cc; i++) {
	    if (FD_ISSET(i, &readfds)) {
		if (i == lsock)
		    new_connection(lsock);
		else if (i == mlsock) {
		    /* -1 means failure, -2 means busy */
		    if ((mcc = connect_meta(mlsock)) >= 0) {
			msock = mcc;
			FD_SET(msock, &writefd_set);
		    }
		} else
		    handle(i);
		cc--;
	    } else if (FD_ISSET(i, &writefds)) {
		if (i == msock) {
		    handle_meta();
		    FD_CLR(msock, &writefd_set);
		    msock = -1;
		} else
		    server_connected(i);
		cc--;
	    }
	}

    }

    /*NOTREACHED*/
}

